import java.io.*;

import java.util.*;

import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;


/**
 * Created by Yann on 10/01/2017.
 */


public class Kmeans {

    private int NUM_CLUSTERS = 3; //Number of Clusters
    private static final int MIN = 5;
    private static final int MAX = 30;
    private static final String FILE_KMEAN = "kmean.json";
    private static final String FILE_VALUES = "values.json";

    private List<Value> values;
    private List<Cluster> clusters;

    public Kmeans() {
        this.values = new ArrayList<Value>();
        this.clusters = new ArrayList<Cluster>();
    }

    protected static double distanceCentroid(double p, double centroid) {
        return Math.abs(centroid - p);
    }


    public void executeKmean()
    {
        init();
        calculate();
        storeKmean();
    }


    public void sortCentroid() {

        Collections.sort(clusters, new Comparator<Cluster>() {
            @Override
            public int compare(Cluster c1, Cluster c2) {
                return Double.compare(c1.getCentroid(), c2.getCentroid());
            }
        });
    }

    public double calculateBatteryConsumptionPerHour(double minutePerCount, double val, long counter)
    {

        double result = (val / (counter * minutePerCount)) * 60; //calcul du pourcentage de batterie consomme en 1h
        System.out.println("BatteryConsumption : "  + result + " with counter " + counter + " and value : " + val);
        return result;

    }

    public void readValues() {
        try {
            // read the json file
            FileReader reader = new FileReader(FILE_VALUES);

            JSONParser jsonParser = new JSONParser();
            JSONObject jsonObject = (JSONObject) jsonParser.parse(reader);


            // get an array from the JSON object
            JSONArray info = (JSONArray) jsonObject.get("yann");
            Iterator i = info.iterator();

            // take each value from the json array separately
            while (i.hasNext()) {
                JSONObject innerObj = (JSONObject) i.next();
                double batteryConsumption = calculateBatteryConsumptionPerHour(15,(double) innerObj.get("Battery_consumption"),(long) innerObj.get("Counter"));
                Value v = new Value(batteryConsumption, (double) innerObj.get("Latitude"), (double) innerObj.get("Longitude"));
                values.add(v);
            }

        } catch (FileNotFoundException ex) {
            ex.printStackTrace();
        } catch (IOException ex) {
            ex.printStackTrace();
        } catch (ParseException ex) {
            ex.printStackTrace();
        } catch (NullPointerException ex) {
            ex.printStackTrace();
        }
        System.out.println("length " + values.size());

        for (Value v : values) {

            System.out.println(v.toString());
        }


    }

    public void storeKmean() {

        sortCentroid();
        System.out.println();


        try (FileWriter fw = new FileWriter(FILE_KMEAN, false);
             BufferedWriter bw = new BufferedWriter(fw);
             PrintWriter out = new PrintWriter(bw)) {
            out.println("{");

           /*First cluster*/
            out.println("\"cluster0\" : {");
            List<Value> cluster0Values = clusters.get(0).getValues();
            out.println("\"centroid\" : " + clusters.get(0).getCentroid() + ",");
            out.println("\"values\" : [{");

            if (cluster0Values.size() > 0) {
                for (int i = 0; i < cluster0Values.size() - 1; i++) {
                    out.println("\"Latitude\" : " + cluster0Values.get(i).getLatitude() + ",");
                    out.println("\"Longitude\" : " + cluster0Values.get(i).getLongitude() + ",");
                    out.println("\"Battery_consumption\" : " + cluster0Values.get(i).getVal());
                    out.println("},\n{");
                }
                out.println("\"Latitude\" : " + cluster0Values.get(cluster0Values.size() - 1).getLatitude() + ",");
                out.println("\"Longitude\" : " + cluster0Values.get(cluster0Values.size() - 1).getLongitude() + ",");
                out.println("\"Battery_consumption\" : " + cluster0Values.get(cluster0Values.size() - 1).getVal());
            }
            out.println("}]");
            out.println("},");


            /*Second cluster*/

            out.println("\"cluster1\" : {");
            out.println("\"centroid\" : " + clusters.get(1).getCentroid() + ",");
            out.println("\"values\" : [{");

            List<Value> cluster1Values = clusters.get(1).getValues();
            if (cluster1Values.size() > 0) {
                for (int i = 0; i < cluster1Values.size() - 1; i++) {
                    out.println("\"Latitude\" : " + cluster1Values.get(i).getLatitude() + ",");
                    out.println("\"Longitude\" : " + cluster1Values.get(i).getLongitude() + ",");
                    out.println("\"Battery_consumption\" : " + cluster1Values.get(i).getVal());
                    out.println("},\n{");
                }
                out.println("\"Latitude\" : " + cluster1Values.get(cluster1Values.size() - 1).getLatitude() + ",");
                out.println("\"Longitude\" : " + cluster1Values.get(cluster1Values.size() - 1).getLongitude() + ",");
                out.println("\"Battery_consumption\" : " + cluster1Values.get(cluster1Values.size() - 1).getVal());
            }
            out.println("}]");
            out.println("},");

             /*Third cluster*/

            out.println("\"cluster2\" : {");
            out.println("\"centroid\" : " + clusters.get(2).getCentroid() + ",");
            out.println("\"values\" : [{");

            List<Value> cluster2Values = clusters.get(2).getValues();
            if (cluster2Values.size() > 0) {
                for (int i = 0; i < cluster2Values.size() - 1; i++) {
                    out.println("\"Latitude\" : " + cluster2Values.get(i).getLatitude() + ",");
                    out.println("\"Longitude\" : " + cluster2Values.get(i).getLongitude() + ",");
                    out.println("\"Battery_consumption\" : " + cluster2Values.get(i).getVal());
                    out.println("},\n{");
                }
                out.println("\"Latitude\" : " + cluster2Values.get(cluster2Values.size() - 1).getLatitude() + ",");
                out.println("\"Longitude\" : " + cluster2Values.get(cluster2Values.size() - 1).getLongitude() + ",");
                out.println("\"Battery_consumption\" : " + cluster2Values.get(cluster2Values.size() - 1).getVal());
            }
            out.println("}]");
            out.println("}");


            out.println("}");


        } catch (IOException e) {

        }
    }

    //Initializes the process
    public void init() {


        this.values = new ArrayList<Value>();
        this.clusters = new ArrayList<Cluster>();

        readValues(); // get the values from json file

        //Create Clusters
        //Set Random Centroids
        for (int i = 0; i < NUM_CLUSTERS; i++) {
            Cluster cluster = new Cluster(i);
            Random r = new Random();
            double centroid = MIN + (MAX - MIN) * r.nextDouble();
           // Value centroid = Value.createRandomValue(MIN, MAX);
            cluster.setCentroid(centroid);
            clusters.add(cluster);
        }

        //Print Initial state
        System.out.println("Initial state : \n");
        plotClusters();
        System.out.println("All the generated Points: \n");
        for (Value p : values) {
            System.out.println(p);
        }
        System.out.println("\n");

    }

    private void plotClusters() {
        for (int i = 0; i < NUM_CLUSTERS; i++) {
            Cluster c = clusters.get(i);
            c.plotCluster();
        }
    }

    //The process to calculate the K Means, with iterating method.
    public void calculate() {
        boolean finish = false;
        int iteration = 0;

        // Add in new data, one at a time, recalculating centroids with each new one.
        while (!finish) {
            //Clear cluster state
            clearClusters();
            List lastCentroids = getCentroids();

            //Assign values to the closer cluster
            assignCluster();
            //Calculate new centroids.
            calculateCentroids();
            iteration++;
            List currentCentroids = getCentroids();

            //Calculates total distance between new and old Centroids
            double distance = 0;
            for (int i = 0; i < lastCentroids.size(); i++) {
                distance += distanceCentroid((double)lastCentroids.get(i), (double)currentCentroids.get(i));
            }
            System.out.println("#################");
            System.out.println("Iteration: " + iteration);
            System.out.println("Centroid distances: " + distance);
            plotClusters();

            if (distance == 0) {
                finish = true;
            }
        }
    }

    private void clearClusters() {
        for (Cluster cluster : clusters) {
            cluster.clear();
        }
    }

    private List getCentroids() {
        List centroids = new ArrayList(NUM_CLUSTERS);
        for (Cluster cluster : clusters) {
            double aux = cluster.getCentroid();
            centroids.add(aux);
        }
        return centroids;
    }

    private void assignCluster() {
        double max = Double.MAX_VALUE;
        double min = max;
        int cluster = 0;
        double distance = 0.0;

        for (Value value : values) {
            min = max;
            for (int i = 0; i < NUM_CLUSTERS; i++) {
                Cluster c = clusters.get(i);
                distance = Value.distance(value, c.getCentroid());
                if (distance < min) {
                    min = distance;
                    cluster = i;
                }
            }
            value.setCluster(cluster);
            clusters.get(cluster).addValue(value);
        }
    }

    private void calculateCentroids() {
        for (Cluster cluster : clusters) {
            double sumX = 0;
            List<Value> list = cluster.getValues();
            int n_points = list.size();

            for (Value value : list) {
                sumX += value.getVal();
            }

            double centroid = cluster.getCentroid();
            if (n_points > 0) {
                double newX = sumX / n_points;
                cluster.setCentroid(newX);
            }
        }
    }
}
